from __future__ import annotations

import dataclasses
import platform
import shutil
import subprocess
import sys
from datetime import datetime
from typing import Dict, List, Optional

try:
    import psutil  # type: ignore
except ImportError:  # pragma: no cover - psutil should be installed with requirements
    psutil = None


@dataclasses.dataclass
class SystemOverview:
    computer_name: str
    os_version: str
    architecture: str
    cpu: str
    physical_cores: Optional[int]
    logical_cores: Optional[int]
    base_clock_ghz: Optional[float]
    total_ram_gb: Optional[float]
    available_ram_gb: Optional[float]
    boot_time: Optional[datetime]
    uptime: Optional[str]
    system_drive_free_gb: Optional[float]


def _format_uptime(boot_time: Optional[float]) -> Optional[str]:
    if not boot_time:
        return None
    delta = datetime.now() - datetime.fromtimestamp(boot_time)
    days = delta.days
    hours, remainder = divmod(delta.seconds, 3600)
    minutes, _ = divmod(remainder, 60)
    return f"{days}d {hours}h {minutes}m"


def _get_cpu_name() -> str:
    try:
        output = subprocess.check_output(
            ["wmic", "cpu", "get", "Name"], stderr=subprocess.DEVNULL, text=True
        )
        lines = [line.strip() for line in output.splitlines() if line.strip()]
        return lines[1] if len(lines) > 1 else platform.processor() or "Unknown CPU"
    except (subprocess.SubprocessError, FileNotFoundError):
        return platform.processor() or "Unknown CPU"


def collect_system_overview() -> SystemOverview:
    boot_timestamp: Optional[float] = None
    total_ram_gb: Optional[float] = None
    available_ram_gb: Optional[float] = None
    logical_cores: Optional[int] = None
    physical_cores: Optional[int] = None

    if psutil:
        try:
            boot_timestamp = psutil.boot_time()
            mem = psutil.virtual_memory()
            total_ram_gb = round(mem.total / (1024**3), 2)
            available_ram_gb = round(mem.available / (1024**3), 2)
            logical_cores = psutil.cpu_count(logical=True)
            physical_cores = psutil.cpu_count(logical=False)
        except Exception:
            pass

    system_drive = shutil.disk_usage("C:\\") if sys.platform == "win32" else None
    free_disk = round(system_drive.free / (1024**3), 2) if system_drive else None

    cpu_name = _get_cpu_name()
    clock_mhz = None
    if psutil:
        try:
            freq = psutil.cpu_freq()
            if freq and freq.max:
                clock_mhz = freq.max
        except Exception:
            pass

    base_clock_ghz = round(clock_mhz / 1000, 2) if clock_mhz else None

    return SystemOverview(
        computer_name=platform.node(),
        os_version=f"{platform.system()} {platform.release()} ({platform.version()})",
        architecture=" ".join(platform.architecture()),
        cpu=cpu_name,
        physical_cores=physical_cores,
        logical_cores=logical_cores,
        base_clock_ghz=base_clock_ghz,
        total_ram_gb=total_ram_gb,
        available_ram_gb=available_ram_gb,
        boot_time=datetime.fromtimestamp(boot_timestamp) if boot_timestamp else None,
        uptime=_format_uptime(boot_timestamp),
        system_drive_free_gb=free_disk,
    )


@dataclasses.dataclass
class ProcessInsight:
    pid: int
    name: str
    cpu_percent: float
    memory_usage_mb: float
    status: str


def list_top_processes(limit: int = 5) -> List[ProcessInsight]:
    insights: List[ProcessInsight] = []
    if not psutil:
        return insights

    try:
        # First call cpu_percent with interval=0 to prime values
        for proc in psutil.process_iter(attrs=["pid", "name", "status"]):
            proc.cpu_percent(interval=None)

        for proc in psutil.process_iter(attrs=["pid", "name", "status"]):
            try:
                cpu = proc.cpu_percent(interval=None)
                memory_info = proc.memory_info()
                insights.append(
                    ProcessInsight(
                        pid=proc.info["pid"],
                        name=proc.info["name"] or "Unknown",
                        cpu_percent=round(cpu, 2),
                        memory_usage_mb=round(memory_info.rss / (1024**2), 2),
                        status=proc.info.get("status", "unknown"),
                    )
                )
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                continue
    except Exception:
        return insights

    insights.sort(key=lambda item: (item.cpu_percent, item.memory_usage_mb), reverse=True)
    return insights[:limit]


def summarize_services() -> Dict[str, List[str]]:
    summary: Dict[str, List[str]] = {"running": [], "stopped": [], "disabled": []}
    if not psutil or not hasattr(psutil, "win_service_iter"):
        return summary

    try:
        for svc in psutil.win_service_iter():
            status = svc.status().lower()
            display_name = svc.display_name()
            if status == "running":
                summary["running"].append(display_name)
            elif status == "stopped":
                summary["stopped"].append(display_name)
            else:
                summary["disabled"].append(display_name)
    except Exception:
        return summary

    for key in summary:
        summary[key].sort()
    return summary

